/*
 * Decompiled with CFR 0.152.
 */
package lemmini.tools;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AccessDeniedException;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import lemmini.tools.ToolBox;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

public class CaseInsensitiveFileTree {
    private static final FileNameComparator FILE_NAME_COMPARATOR = new FileNameComparator();
    private final Path root;
    private final Map<String, List<Path>> files = new LinkedHashMap<String, List<Path>>(1024);

    public CaseInsensitiveFileTree(Path directory) throws IOException {
        this(directory, Integer.MAX_VALUE);
    }

    public CaseInsensitiveFileTree(Path directory, int maxDepth) throws IOException {
        this.root = directory;
        this.refresh(maxDepth);
    }

    public final void refresh(int maxDepth) throws IOException {
        this.files.clear();
        final TreeMap filesTemp = new TreeMap(FILE_NAME_COMPARATOR);
        SimpleFileVisitor<Path> visitor = new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                this.addPath(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                this.addPath(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                if (exc instanceof AccessDeniedException) {
                    return FileVisitResult.CONTINUE;
                }
                throw exc;
            }

            private void addPath(Path path) {
                String relativePathStr = CaseInsensitiveFileTree.pathToString(CaseInsensitiveFileTree.this.root.relativize(path));
                List pathVariants = filesTemp.computeIfAbsent(relativePathStr, s -> new ArrayList(1));
                pathVariants.add(path);
            }
        };
        if (Files.exists(this.root, new LinkOption[0])) {
            Files.walkFileTree(this.root, Collections.emptySet(), maxDepth, (FileVisitor<? super Path>)visitor);
        }
        this.files.putAll(filesTemp);
    }

    public Path getRoot() {
        return this.root;
    }

    public Path getPath(String fileName) {
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        Path possiblePath = this.getPath1(normalizedFileName);
        if (possiblePath == null) {
            if (normalizedFileName.equals("/")) {
                return this.root;
            }
            String parent = ToolBox.getParent(fileName);
            String file = ToolBox.getFileName(fileName);
            Path parentPath = this.getPath(parent);
            Path rootPath = this.root.resolve(parentPath);
            Path filePath = rootPath.resolve(file);
            return filePath;
        }
        return possiblePath;
    }

    private Path getPath1(String normalizedFileName) {
        List fileVariants = this.files.getOrDefault(normalizedFileName, Collections.emptyList());
        Iterator it = fileVariants.iterator();
        while (it.hasNext()) {
            Path path = (Path)it.next();
            if (Files.notExists(path, new LinkOption[0])) {
                it.remove();
                continue;
            }
            return path;
        }
        this.files.remove(normalizedFileName);
        return null;
    }

    public List<Path> getAllPaths(String fileName) {
        return Collections.unmodifiableList(this.files.getOrDefault(CaseInsensitiveFileTree.normalize(fileName), Collections.emptyList()));
    }

    public List<Path> getAllPathsRegex(String regex) {
        Pattern p = Pattern.compile(regex, 66);
        return this.files.entrySet().stream().filter(entry -> p.matcher((CharSequence)entry.getKey()).matches()).map(Map.Entry::getValue).collect(() -> new ArrayList(512), List::addAll, List::addAll);
    }

    public boolean exists(String fileName) {
        return !this.files.getOrDefault(CaseInsensitiveFileTree.normalize(fileName), Collections.emptyList()).isEmpty();
    }

    public OutputStream newOutputStream(String fileName, OpenOption ... options) throws IOException {
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        if (CaseInsensitiveFileTree.isDirectory(normalizedFileName)) {
            throw new IOException("Not a valid file name: " + fileName);
        }
        Path possiblePath = this.getPath1(normalizedFileName);
        if (possiblePath == null) {
            String parent = ToolBox.getParent(normalizedFileName);
            possiblePath = this.getPath1(parent);
            if (possiblePath == null) {
                throw new IOException("Directory does not exist: " + ToolBox.getParent(fileName));
            }
            Path path = possiblePath.resolve(ToolBox.getFileName(fileName));
            OutputStream out = Files.newOutputStream(path, options);
            try {
                List pathVariants = this.files.computeIfAbsent(normalizedFileName, s -> new ArrayList(1));
                pathVariants.add(path);
                return out;
            }
            catch (Exception ex) {
                try {
                    out.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw ex;
            }
        }
        return Files.newOutputStream(possiblePath, options);
    }

    public BufferedWriter newBufferedWriter(String fileName, Charset cs, OpenOption ... options) throws IOException {
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        if (CaseInsensitiveFileTree.isDirectory(normalizedFileName)) {
            throw new IOException("Not a valid file name: " + fileName);
        }
        Path possiblePath = this.getPath1(normalizedFileName);
        if (possiblePath == null) {
            String parent = ToolBox.getParent(normalizedFileName);
            possiblePath = this.getPath1(parent);
            if (possiblePath == null) {
                throw new IOException("Directory does not exist: " + ToolBox.getParent(fileName));
            }
            Path path = possiblePath.resolve(ToolBox.getFileName(fileName));
            BufferedWriter w = Files.newBufferedWriter(path, cs, options);
            try {
                List pathVariants = this.files.computeIfAbsent(normalizedFileName, s -> new ArrayList(1));
                pathVariants.add(path);
                return w;
            }
            catch (Exception ex) {
                try {
                    w.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                throw ex;
            }
        }
        return Files.newBufferedWriter(possiblePath, cs, options);
    }

    public BufferedWriter newBufferedWriter(String fileName, OpenOption ... options) throws IOException {
        return this.newBufferedWriter(fileName, StandardCharsets.UTF_8, options);
    }

    public Path createDirectories(String fileName, FileAttribute<?> ... attrs) throws IOException {
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        if (!CaseInsensitiveFileTree.isDirectory(normalizedFileName)) {
            throw new IOException("Not a valid directory name: " + fileName);
        }
        Path directory = this.getPath1(normalizedFileName);
        if (directory == null) {
            if (normalizedFileName.equals("/")) {
                directory = this.root;
            } else {
                Path parent = this.createDirectories(ToolBox.getParent(normalizedFileName), attrs);
                directory = parent.resolve(ToolBox.getFileName(fileName));
            }
            Files.createDirectories(directory, new FileAttribute[0]);
            List directoryVariants = this.files.computeIfAbsent(normalizedFileName, s -> new ArrayList(1));
            directoryVariants.add(directory);
        }
        return directory;
    }

    public void delete(String fileName) throws IOException {
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        if (CaseInsensitiveFileTree.isDirectory(normalizedFileName)) {
            Iterator<Map.Entry<String, List<Path>>> it = this.files.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry3 = it.next();
                String entryName = entry3.getKey();
                if (!entryName.startsWith(normalizedFileName) || CaseInsensitiveFileTree.isDirectory(entryName)) continue;
                List<Path> fileVariants = entry3.getValue();
                Iterator<Path> it2 = fileVariants.iterator();
                while (it2.hasNext()) {
                    Path path = it2.next();
                    Files.deleteIfExists(path);
                    it2.remove();
                }
                it.remove();
            }
            List directoryList = this.files.entrySet().stream().filter(entry -> {
                String entryName = (String)entry.getKey();
                return CaseInsensitiveFileTree.isDirectory(entryName) && entryName.startsWith(normalizedFileName);
            }).sorted((entry1, entry2) -> {
                int slashCount2;
                String entryName1 = (String)entry1.getKey();
                String entryName2 = (String)entry2.getKey();
                int slashCount1 = StringUtils.countMatches((CharSequence)entryName1, (char)'/');
                if (slashCount1 != (slashCount2 = StringUtils.countMatches((CharSequence)entryName2, (char)'/'))) {
                    return slashCount2 - slashCount1;
                }
                return entryName1.compareTo(entryName2);
            }).collect(Collectors.toList());
            for (Map.Entry entry3 : directoryList) {
                String entryName = (String)entry3.getKey();
                List fileVariants = (List)entry3.getValue();
                Iterator it2 = fileVariants.iterator();
                while (it2.hasNext()) {
                    Path path = (Path)it2.next();
                    Files.deleteIfExists(path);
                    it2.remove();
                }
                this.files.remove(entryName);
            }
        } else {
            for (Path path : this.files.getOrDefault(normalizedFileName, Collections.emptyList())) {
                Files.deleteIfExists(path);
            }
            this.files.remove(normalizedFileName);
        }
    }

    public void deleteIfEmpty(String fileName) throws IOException {
        List<Path> fileVariants;
        String normalizedFileName = CaseInsensitiveFileTree.normalize(fileName);
        if (CaseInsensitiveFileTree.isDirectory(normalizedFileName) && (fileVariants = this.files.get(normalizedFileName)) != null) {
            Iterator<Path> it = fileVariants.iterator();
            while (it.hasNext()) {
                Path path = it.next();
                try {
                    Files.deleteIfExists(path);
                    it.remove();
                }
                catch (DirectoryNotEmptyException directoryNotEmptyException) {
                    // empty catch block
                }
            }
            if (fileVariants.isEmpty()) {
                this.files.remove(normalizedFileName);
            }
        }
    }

    private static String pathToString(Path path) {
        StringBuilder sb = new StringBuilder(64);
        for (Path name : path) {
            sb.append(name.toString().toLowerCase(Locale.ROOT)).append('/');
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    private static String normalize(String fileName) {
        if (fileName.isEmpty()) {
            return "/";
        }
        return FilenameUtils.normalize(fileName, true).toLowerCase(Locale.ROOT);
    }

    private static boolean isDirectory(String fileName) {
        return fileName.endsWith("/");
    }

    private static class FileNameComparator
    implements Comparator<String> {
        private FileNameComparator() {
        }

        @Override
        public int compare(String o1, String o2) {
            if (o1.isEmpty() && !o2.isEmpty() || o1.equals("/") && !o2.equals("/")) {
                return -1;
            }
            if (!o1.isEmpty() && o2.isEmpty() || !o1.equals("/") && o2.equals("/")) {
                return 1;
            }
            int length1 = o1.length();
            int length2 = o2.length();
            int minLength = Math.min(length1, length2);
            int i = 0;
            while (i < minLength) {
                char c1 = o1.charAt(i);
                char c2 = o2.charAt(i);
                if (c1 == '/' && c2 != '/') {
                    if (o2.indexOf(47, i) != -1) {
                        return -1;
                    }
                    return 1;
                }
                if (c1 != '/' && c2 == '/') {
                    if (o1.indexOf(47, i) != -1) {
                        return 1;
                    }
                    return -1;
                }
                if (c1 != c2) {
                    int slashPos1 = o1.indexOf(47, i);
                    int slashPos2 = o2.indexOf(47, i);
                    if (slashPos1 == -1 && slashPos2 != -1) {
                        return -1;
                    }
                    if (slashPos1 != -1 && slashPos2 == -1) {
                        return 1;
                    }
                    return c1 - c2;
                }
                ++i;
            }
            return length1 - length2;
        }
    }
}

